/*
 * Simple test harness to check workings for calculating safe nude sunbathing
 * areas in the garden.
 * 
 * nib 2007-04-24
 */

#define DEBUG 0

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define sgn(x) ((int)(((x) < 0) ? -1 : ((x) > 0)? 1 : 0))

/*
 * Most model data are global. All XY co-ordinates in this model are in units
 * of the array index (which happen to represent 0.5m at the moment). Where
 * necessary we imagine the centre points of the cells to be at 0.5-cell
 * values and the boundaries at whole integers. Index [0][0] is the NE
 * corner of the patio, [26][0] is the SE corner by the willow. The Z coord
 * is read in as metres but is converted to 0.5m units.
 */
struct
{
	double z;
	int fence;
	int visible;
	int shadow;
}
garden[22][26];

void bomb(long n, char *mess)
{
	fprintf(stderr, "%ld, %s\n", mess);
}

void load_simple_data(void)
{
	FILE *fip;
	long i, j;
	char s[100];

	fip = fopen("sdata.txt", "r");
	if (fip == NULL) bomb (1, "can't open garden data");

	for (i=0; i<26; i++)
		for (j=0; j<22; j++)
		{
			if (NULL == fgets(s, 100, fip))
				bomb(2, "error in reading garden data");
			garden[j][25-i].z = 2.0 * atof(s);
			garden[j][25-i].fence = 0;
			garden[j][25-i].visible = 0;
			garden[j][25-i].shadow = 0;
		}
}

void test_print_garden(void)
{
	long i, j;

	for (i=0; i<26; i++)
	{
		for (j=0; j<22; j++)
		{
			printf("%lf ", garden[j][i].z);
		}
		printf("\n");
	}
}

void print_result(void)
{
	long i, j;
	char c;

	for (i=0; i<26; i++)
	{
		for (j=0; j<22; j++)
		{
			if (garden[j][25-i].fence)
				c = '*';
			else if (garden[j][25-i].visible)
				c = 'O';
			else if (garden[j][25-i].shadow)
				c = 'S';
			else
				c = ' ';
			printf("%c ", c);
		}
		printf("\n");
	}
}

/*
 * Test routine goes round each cell, adds the required clear height,
 * draws a line from here to a specified observer point, goes along the
 * line working out which other cells it crosses, if the height of any
 * other cell is greater than the line height, makes the initial cell
 * not clear and gets on with the next initial cell.
 */
void update(double xa, double ya, double za)
{
	double xe, ye, ze, xc, yc, zc, xp, yp, xx, yy, zz;
	int x, y, xdir, ydir, blocked, xv, yv;

	xe = xa*2; ye = ya*2; ze = za*2;	/* metres to interval */
	for (y=0; y<26; y++) for (x=0; x<22; x++)
	{
		/* mark the fences, anything over 0.2m */
		if (garden[x][y].z >= 1 )
		{
			garden[x][y].fence = 1;
			continue;
		}

		/* required height, above centre of cell */
		xc = x + 0.5;
		yc = y + 0.5;
		zc = garden[x][y].z + 1; /* 0.5m up for sunbathing */

		/* so the line is xc,yc,zc to xe,ye,ze */
		xdir = sgn(xe - xc);
		ydir = sgn(ye - yc);
		/* we only go in these directions, until we hit something
		 * for +ve xdir we look at x values from xc+0.5 to 25
		 * for -ve xdir we look at x values from xc-0.5 to 0
		 * for +ve ydir we look at y values from yc+0.5 to 21
		 * for -ve ydir we look at y values from yc-0.5 to 0
		 * in each case take floor() of the xy and this is the
		 * address of a cell that we cross.
		 */

		/* scan in x */
		blocked = 0;
		xp = xc + 0.5;
		while (xdir >= 0 && xp < 22 && xp <= xe)
		{
			/* co-ords of line intersecting plane x=xp */
			yy = (ye-yc)/(xe-xc) * (xp-xc) + yc;
			zz = (ze-zc)/(xe-xc) * (xp-xc) + zc;
			/* square crossed */
			xv = (int)xp;
			yv = (int)floor(yy);
			/* do we hit? */
			if (yv <= 25 && yv >= 0)
				if (garden[xv][yv].z > zz) blocked++;
#if DEBUG
printf("+x at %d,%d over %d %d z=%lf block=%d\n", x, y, xv, yv, zz, blocked);
#endif
			xp++;
		}
		xp = xc - 0.5;
		while (xdir < 0 && xp >= 0 && xp >= xe)
		{
			/* co-ords of line intersecting plane x=xp */
			yy = (ye-yc)/(xe-xc) * (xp-xc) + yc;
			zz = (ze-zc)/(xe-xc) * (xp-xc) + zc;
			/* square crossed */
			xv = (int)xp;
			yv = (int)floor(yy);
			/* do we hit? */
			if (yv <= 25 && yv >= 0)
				if (garden[xv][yv].z > zz) blocked++;
#if DEBUG
printf("-x at %d,%d over %d %d z=%lf block=%d\n", x, y, xv, yv, zz, blocked);
#endif
			xp--;
		}

		/* scan in y */
		yp = yc + 0.5;
		while (ydir >= 0 && yp < 26 && yp <= ye)
		{
			/* co-ords of line intersecting plane y=yp */
			xx = (xe-xc)/(ye-yc) * (yp-yc) + xc;
			zz = (ze-zc)/(ye-yc) * (yp-yc) + zc;
			/* square crossed */
			xv = (int)floor(xx);
			yv = (int)yp;
			/* do we hit? */
			if (xv <= 21 && xv >= 0)
				if (garden[xv][yv].z > zz) blocked++;
#if DEBUG
printf("+y at %d,%d over %d %d z=%lf block=%d\n", x, y, xv, yv, zz, blocked);
#endif
			yp++;
		}
		yp = yc - 0.5;
		while (ydir < 0 && yp >= 0 && yp >= ye)
		{
			/* co-ords of line intersecting plane y=yp */
			xx = (xe-xc)/(ye-yc) * (yp-yc) + xc;
			zz = (ze-zc)/(ye-yc) * (yp-yc) + zc;
			/* square crossed */
			xv = (int)floor(xx);
			yv = (int)yp;
			/* do we hit? */
			if (xv <= 21 && xv >= 0)
				if (garden[xv][yv].z > zz) blocked++;
#if DEBUG
printf("-y at %d,%d over %d %d z=%lf block=%d\n", x, y, xv, yv, zz, blocked);
#endif
			yp--;
		}

		if (blocked == 0) garden[x][y].visible = 1;
	}
}

/*
 * Same as update except updates the shadow tag if area is in shadow. Sun
 * is approximated elsewhere by converting its AZ and EL into garden
 * coordinates as an object at the correct angles a long way away.
 */
void addsun(double xa, double ya, double za)
{
	double xe, ye, ze, xc, yc, zc, xp, yp, xx, yy, zz;
	int x, y, xdir, ydir, blocked, xv, yv;

	xe = xa*2; ye = ya*2; ze = za*2;	/* metres to interval */
	for (y=0; y<26; y++) for (x=0; x<22; x++)
	{
		/* mark the fences, anything over 0.2m */
		if (garden[x][y].z >= 1 )
		{
			garden[x][y].fence = 1;
			continue;
		}

		/* required height, above centre of cell */
		xc = x + 0.5;
		yc = y + 0.5;
		zc = garden[x][y].z + 1; /* 0.5m up for sunbathing */

		/* so the line is xc,yc,zc to xe,ye,ze */
		xdir = sgn(xe - xc);
		ydir = sgn(ye - yc);
		/* we only go in these directions, until we hit something
		 * for +ve xdir we look at x values from xc+0.5 to 25
		 * for -ve xdir we look at x values from xc-0.5 to 0
		 * for +ve ydir we look at y values from yc+0.5 to 21
		 * for -ve ydir we look at y values from yc-0.5 to 0
		 * in each case take floor() of the xy and this is the
		 * address of a cell that we cross.
		 */

		/* scan in x */
		blocked = 0;
		xp = xc + 0.5;
		while (xdir >= 0 && xp < 22 && xp <= xe)
		{
			/* co-ords of line intersecting plane x=xp */
			yy = (ye-yc)/(xe-xc) * (xp-xc) + yc;
			zz = (ze-zc)/(xe-xc) * (xp-xc) + zc;
			/* square crossed */
			xv = (int)xp;
			yv = (int)floor(yy);
			/* do we hit? */
			if (yv <= 25 && yv >= 0)
				if (garden[xv][yv].z > zz) blocked++;
#if DEBUG
printf("+x at %d,%d over %d %d z=%lf block=%d\n", x, y, xv, yv, zz, blocked);
#endif
			xp++;
		}
		xp = xc - 0.5;
		while (xdir < 0 && xp >= 0 && xp >= xe)
		{
			/* co-ords of line intersecting plane x=xp */
			yy = (ye-yc)/(xe-xc) * (xp-xc) + yc;
			zz = (ze-zc)/(xe-xc) * (xp-xc) + zc;
			/* square crossed */
			xv = (int)xp;
			yv = (int)floor(yy);
			/* do we hit? */
			if (yv <= 25 && yv >= 0)
				if (garden[xv][yv].z > zz) blocked++;
#if DEBUG
printf("-x at %d,%d over %d %d z=%lf block=%d\n", x, y, xv, yv, zz, blocked);
#endif
			xp--;
		}

		/* scan in y */
		yp = yc + 0.5;
		while (ydir >= 0 && yp < 26 && yp <= ye)
		{
			/* co-ords of line intersecting plane y=yp */
			xx = (xe-xc)/(ye-yc) * (yp-yc) + xc;
			zz = (ze-zc)/(ye-yc) * (yp-yc) + zc;
			/* square crossed */
			xv = (int)floor(xx);
			yv = (int)yp;
			/* do we hit? */
			if (xv <= 21 && xv >= 0)
				if (garden[xv][yv].z > zz) blocked++;
#if DEBUG
printf("+y at %d,%d over %d %d z=%lf block=%d\n", x, y, xv, yv, zz, blocked);
#endif
			yp++;
		}
		yp = yc - 0.5;
		while (ydir < 0 && yp >= 0 && yp >= ye)
		{
			/* co-ords of line intersecting plane y=yp */
			xx = (xe-xc)/(ye-yc) * (yp-yc) + xc;
			zz = (ze-zc)/(ye-yc) * (yp-yc) + zc;
			/* square crossed */
			xv = (int)floor(xx);
			yv = (int)yp;
			/* do we hit? */
			if (xv <= 21 && xv >= 0)
				if (garden[xv][yv].z > zz) blocked++;
#if DEBUG
printf("-y at %d,%d over %d %d z=%lf block=%d\n", x, y, xv, yv, zz, blocked);
#endif
			yp--;
		}

		if (blocked != 0) garden[x][y].shadow = 1;
	}
}

int main(int argc, char **argv)
{
	load_simple_data();

#if 1
	update(-4,24,2.8);	/* window 1 backing to Keith */
#endif
#if 1
	update(-2,24,2.8);	/* window 2 backing to Keith */
#endif
#if 1
	update(2,20,2.8);	/* window 3 backing to us */
#endif
#if 1
	update(4,20,2.8);	/* window 4 backing to us */
#endif
#if 1
	update(6,20,2.8);	/* window 5 backing to us */
#endif
#if 1
	update(8,20,2.8);	/* window 6 backing to us */
#endif
#if 1
	update(20,3,4.6);	/* window 7 on #25 */
#endif
#if 1
	update(20,1,4.6);	/* window 8 on #25 */
#endif
#if 0
	update(-1,1,5);		/* window 9 Keith bathroom */
#endif
#if 0
	update(-3,1,5); 	/* window 10 Keith landing */
#endif
#if 1
	update(-5,1,5); 	/* window 11 Keith bedroom */
#endif

#if 0
	addsun(-4536,8912,0);		/* 09h mar/sep */	
	addsun(-1761,8296,5298);	/* 10h mar/sep */
	addsun(426,8080,5877);		/* 11h mar/sep */
	addsun(2437,7495,6156);		/* 12h mar/sep */
	addsun(4351,6698,6017);		/* 13h mar/sep */
	addsun(6234,5611,5445);		/* 14h mar/sep */
	addsun(7940,4045,4539);		/* 15h mar/sep */
	addsun(9782,2079,0);		/* 16h mar/sep */

	addsun(-3670,7210,5877);	/* 09h apr/aug */
	addsun(-1518,7155,6819);	/* 10h apr/aug */
	addsun(352,6683,7430);		/* 11h apr/aug */
	addsun(1988,6114,7659);		/* 12h apr/aug */
	addsun(3646,5612,7430);		/* 13h apr/aug */
	addsun(5436,4894,6819);		/* 14h apr/aug */
	addsun(7209,3673,5877);		/* 15h apr/aug */
	addsun(8637,1835,4694);		/* 16h apr/aug */

	addsun(-3318,6519,6819);	/* 09h may/jul */
	addsun(-1307,6157,7770);	/* 10h may/jul */
	addsun(294,5586,8289);		/* 11h may/jul */
	addsun(1639,5041,8479);		/* 12h may/jul */
	addsun(3047,4690,8289);		/* 13h may/jul */
	addsun(4778,4301,7659);		/* 14h may/jul */
	addsun(6623,3374,6690);		/* 15h may/jul */
	addsun(8110,1723,5591);		/* 16h may/jul */

	addsun(-3208,6303,7070);	/* 09h jun */
	addsun(-1250,5888,7985);	/* 10h jun */
	addsun(271,5145,8571);		/* 11h jun */
	addsun(1500,4612,8745);		/* 12h jun */
	addsun(2807,4320,8571);		/* 13h jun */
	addsun(4370,3933,8089);		/* 14h jun */
	addsun(6191,3154,7192);		/* 15h jun */
	addsun(7813,1660,6017);		/* 16h jun */
#endif

	print_result();
}
